<?php
// ARCHIVO: #6 - app/Http/Middleware/CheckRole.php
// VERSIÓN: UrbanOSS 2.0 - Laravel 11.47
// PREDECESORES: User.php (#1), LoginController.php (#5)
// RUTA EXACTA: app/Http/Middleware/CheckRole.php
// COMPATIBILIDAD: Laravel 11.47, PHP 8.2+, MySQL 5.5+
// REQUISITOS: Control de acceso granular por roles
// FECHA: 2024-12-08

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Route;
use Symfony\Component\HttpFoundation\Response;

class CheckRole
{
    /**
     * Handle an incoming request.
     */
    public function handle(Request $request, Closure $next, ...$roles): Response
    {
        // 1. Verificar autenticación
        if (!Auth::check()) {
            return $this->redirectToLogin($request);
        }

        $user = Auth::user();
        
        // 2. Verificar estado del usuario
        if (!$this->validateUserAccess($user)) {
            return $this->handleInvalidAccess($user);
        }

        // 3. Verificar rol
        if (!$this->userHasRequiredRole($user, $roles)) {
            return $this->handleUnauthorizedAccess($user, $request, $roles);
        }

        // 4. Verificar condominio (excepto superadmin)
        if (!$this->validateCondominioAccess($user)) {
            return $this->handleInvalidCondominio($user);
        }

        // 5. Registrar acceso (solo en modo debug)
        $this->logAccess($user, $request, $roles);

        return $next($request);
    }

    /**
     * Validate user access status.
     */
    protected function validateUserAccess($user): bool
    {
        // Usar métodos del modelo User
        return method_exists($user, 'isActive') ? $user->isActive() : true;
    }

    /**
     * Check if user has required role(s).
     */
    protected function userHasRequiredRole($user, array $roles): bool
    {
        // Superadmin tiene acceso total
        if (property_exists($user, 'role') && $user->role === 'superadmin') {
            return true;
        }

        // Si no hay roles específicos, permitir acceso
        if (empty($roles)) {
            return true;
        }

        // Verificar si el usuario tiene alguno de los roles requeridos
        return in_array($user->role, $roles);
    }

    /**
     * Validate condominio access.
     */
    protected function validateCondominioAccess($user): bool
    {
        // Superadmin no necesita validar condominio
        if ($user->role === 'superadmin') {
            return true;
        }

        // Usuarios sin condominio no pueden acceder (excepto superadmin)
        if (!$user->condominio_id) {
            return false;
        }

        // Verificar relación y estado del condominio
        if (!$user->condominio) {
            Log::warning('Condominio no encontrado para usuario', [
                'user_id' => $user->id,
                'condominio_id' => $user->condominio_id
            ]);
            return false;
        }

        // Verificar si el condominio está activo
        return $user->condominio->estado === 'activo';
    }

    /**
     * Redirect to login with message.
     */
    protected function redirectToLogin(Request $request)
    {
        Session::flash('status', 'warning');
        Session::flash('message', 'Por favor inicie sesión para acceder a esta página.');

        // Guardar URL para redirección después del login
        if (!$request->is('login')) {
            Session::put('url.intended', $request->fullUrl());
        }

        return redirect()->route('login');
    }

    /**
     * Handle invalid user access.
     */
    protected function handleInvalidAccess($user)
    {
        Auth::logout();
        Session::invalidate();

        $message = 'Su cuenta está inactiva o pendiente de aprobación.';
        $status = 'error';

        if (property_exists($user, 'estado')) {
            if ($user->estado === 'pendiente') {
                $message = 'Su cuenta está pendiente de aprobación por el administrador.';
                $status = 'warning';
            } elseif ($user->estado === 'inactivo') {
                $message = 'Su cuenta está inactiva. Contacte al administrador.';
                $status = 'error';
            }
        }

        return redirect()->route('login')
            ->with('status', $status)
            ->with('message', $message);
    }

    /**
     * Handle unauthorized access.
     */
    protected function handleUnauthorizedAccess($user, Request $request, array $roles)
    {
        Log::warning('Acceso no autorizado', [
            'user_id' => $user->id,
            'email' => $user->email,
            'user_role' => $user->role,
            'required_roles' => $roles,
            'ip' => $request->ip(),
            'url' => $request->fullUrl(),
            'user_agent' => $request->userAgent(),
            'timestamp' => now()->toDateTimeString(),
        ]);

        // Para solicitudes AJAX/API
        if ($request->expectsJson()) {
            return response()->json([
                'success' => false,
                'error' => 'unauthorized',
                'message' => 'No tiene permisos para acceder a este recurso.',
                'required_roles' => $roles,
                'user_role' => $user->role,
            ], 403);
        }

        // Para peticiones web
        if (view()->exists('auth.unauthorized')) {
            return response()->view('auth.unauthorized', [
                'required_roles' => $roles,
                'user_role' => $user->role,
            ], 403);
        }

        // Fallback: redirigir a login con mensaje
        Session::flash('status', 'error');
        Session::flash('message', 'No tiene permisos para acceder a esta sección.');

        return redirect()->route('login');
    }

    /**
     * Handle invalid condominio access.
     */
    protected function handleInvalidCondominio($user)
    {
        Auth::logout();
        Session::invalidate();

        Log::error('Acceso a condominio inválido', [
            'user_id' => $user->id,
            'email' => $user->email,
            'condominio_id' => $user->condominio_id,
            'condominio_estado' => $user->condominio->estado ?? 'no encontrado',
            'timestamp' => now()->toDateTimeString(),
        ]);

        return redirect()->route('login')
            ->with('status', 'error')
            ->with('message', 'El condominio asociado está inactivo o no disponible.');
    }

    /**
     * Log successful access.
     */
    protected function logAccess($user, Request $request, array $roles): void
    {
        // Solo en desarrollo para evitar llenar logs
        if (config('app.debug') && config('app.env') === 'local') {
            Log::debug('Acceso autorizado', [
                'user_id' => $user->id,
                'email' => $user->email,
                'role' => $user->role,
                'condominio_id' => $user->condominio_id,
                'required_roles' => $roles,
                'route' => Route::currentRouteName(),
                'ip' => $request->ip(),
                'timestamp' => now()->toDateTimeString(),
            ]);
        }
    }
}