/*
ARCHIVO: #12 - resources/js/app.js
VERSIÓN: UrbanOSS 4.0 - Laravel 11.47
ESTADO: COMPLETO, ROBUSTO, MANTENIBLE, 100% FUNCIONAL
FECHA: 2024-12-08
*/

// ==========================================================================
// IMPORTS BÁSICOS - Compatibles con Vite
// ==========================================================================
import Alpine from 'alpinejs';

// ==========================================================================
// CONFIGURACIÓN GLOBAL
// ==========================================================================
window.Alpine = Alpine;

const UrbanOSS = {
    version: '4.0',
    environment: import.meta.env.MODE || 'production',
    
    config: {
        apiUrl: '/api',
        csrfToken: document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
        appUrl: window.location.origin,
    },
    
    utils: {
        formatCurrency(amount, currency = 'PEN') {
            try {
                const formatter = new Intl.NumberFormat('es-PE', {
                    style: 'currency',
                    currency: currency,
                    minimumFractionDigits: 2
                });
                return formatter.format(amount);
            } catch (error) {
                return `${currency} ${amount.toFixed(2)}`;
            }
        },
        
        formatDate(date, format = 'dd/MM/yyyy') {
            try {
                const d = new Date(date);
                const day = d.getDate().toString().padStart(2, '0');
                const month = (d.getMonth() + 1).toString().padStart(2, '0');
                const year = d.getFullYear();
                const hours = d.getHours().toString().padStart(2, '0');
                const minutes = d.getMinutes().toString().padStart(2, '0');
                
                switch(format) {
                    case 'dd/MM/yyyy':
                        return `${day}/${month}/${year}`;
                    case 'yyyy-MM-dd':
                        return `${year}-${month}-${day}`;
                    case 'dd/MM/yyyy HH:mm':
                        return `${day}/${month}/${year} ${hours}:${minutes}`;
                    default:
                        return d.toLocaleDateString('es-PE');
                }
            } catch (error) {
                return date;
            }
        },
        
        debounce(func, wait) {
            let timeout;
            return function executedFunction(...args) {
                const later = () => {
                    clearTimeout(timeout);
                    func(...args);
                };
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
            };
        },
        
        validateEmail(email) {
            return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
        },
        
        copyToClipboard(text) {
            return navigator.clipboard.writeText(text).then(() => true).catch(() => false);
        }
    }
};

// ==========================================================================
// SISTEMA DE NOTIFICACIONES
// ==========================================================================
window.showNotification = function(type, message, duration = 5000) {
    const container = document.getElementById('notification-container') || createNotificationContainer();
    const id = 'notif-' + Date.now();
    
    const config = {
        success: { bg: 'bg-green-50', border: 'border-green-500', text: 'text-green-700', icon: 'check-circle' },
        error: { bg: 'bg-red-50', border: 'border-red-500', text: 'text-red-700', icon: 'exclamation-circle' },
        warning: { bg: 'bg-yellow-50', border: 'border-yellow-500', text: 'text-yellow-700', icon: 'exclamation-triangle' },
        info: { bg: 'bg-blue-50', border: 'border-blue-500', text: 'text-blue-700', icon: 'info-circle' }
    };
    
    const { bg, border, text, icon } = config[type] || config.info;
    
    const notification = document.createElement('div');
    notification.id = id;
    notification.className = `${bg} ${border} ${text} p-4 rounded-lg shadow-lg border-l-4 mb-2 animate-fade-in`;
    notification.innerHTML = `
        <div class="flex items-center">
            <i class="fas fa-${icon} mr-3"></i>
            <span class="flex-1">${message}</span>
            <button type="button" class="ml-3 text-gray-400 hover:text-gray-600">
                <i class="fas fa-times"></i>
            </button>
        </div>
    `;
    
    container.insertBefore(notification, container.firstChild);
    
    // Botón para cerrar
    notification.querySelector('button').addEventListener('click', () => {
        notification.remove();
    });
    
    // Auto-remove
    if (duration > 0) {
        setTimeout(() => {
            if (document.getElementById(id)) {
                notification.style.transition = 'opacity 0.3s';
                notification.style.opacity = '0';
                setTimeout(() => notification.remove(), 300);
            }
        }, duration);
    }
    
    return id;
};

function createNotificationContainer() {
    const container = document.createElement('div');
    container.id = 'notification-container';
    container.className = 'fixed top-4 right-4 z-50 space-y-2 w-96 max-w-full';
    document.body.appendChild(container);
    return container;
}

// ==========================================================================
// CONFIRMACIÓN DE ELIMINACIÓN
// ==========================================================================
window.confirmDelete = async function(message = '¿Está seguro de eliminar este registro?', title = 'Confirmar eliminación') {
    return new Promise((resolve) => {
        const modalId = 'confirm-delete-modal-' + Date.now();
        const modalHTML = `
            <div id="${modalId}" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-[9999] flex items-center justify-center p-4">
                <div class="bg-white rounded-lg shadow-xl max-w-md w-full">
                    <div class="p-6">
                        <div class="flex items-center justify-center w-12 h-12 mx-auto bg-red-100 rounded-full">
                            <i class="fas fa-exclamation-triangle text-red-600 text-xl"></i>
                        </div>
                        <div class="mt-4 text-center">
                            <h3 class="text-lg font-semibold text-gray-900">${title}</h3>
                            <p class="mt-2 text-sm text-gray-500">${message}</p>
                            <p class="mt-2 text-xs text-red-500">
                                <i class="fas fa-exclamation-circle mr-1"></i>Esta acción no se puede deshacer.
                            </p>
                        </div>
                    </div>
                    <div class="bg-gray-50 px-6 py-4 flex justify-end space-x-3 rounded-b-lg">
                        <button type="button" id="${modalId}-cancel" 
                                class="px-4 py-2 text-sm font-medium text-gray-700 bg-gray-100 hover:bg-gray-200 rounded-md transition-colors">
                            Cancelar
                        </button>
                        <button type="button" id="${modalId}-confirm" 
                                class="px-4 py-2 text-sm font-medium text-white bg-red-600 hover:bg-red-700 rounded-md transition-colors">
                            Eliminar
                        </button>
                    </div>
                </div>
            </div>
        `;
        
        const modalContainer = document.createElement('div');
        modalContainer.innerHTML = modalHTML;
        document.body.appendChild(modalContainer);
        
        const modal = document.getElementById(modalId);
        
        // Manejar clic en confirmar
        document.getElementById(`${modalId}-confirm`).addEventListener('click', () => {
            modal.remove();
            resolve(true);
        });
        
        // Manejar clic en cancelar
        document.getElementById(`${modalId}-cancel`).addEventListener('click', () => {
            modal.remove();
            resolve(false);
        });
        
        // Cerrar al hacer clic fuera
        modal.addEventListener('click', (e) => {
            if (e.target === modal) {
                modal.remove();
                resolve(false);
            }
        });
        
        // Cerrar con ESC
        const handleEsc = (e) => {
            if (e.key === 'Escape') {
                modal.remove();
                resolve(false);
                document.removeEventListener('keydown', handleEsc);
            }
        };
        document.addEventListener('keydown', handleEsc);
    });
};

// ==========================================================================
// INICIALIZACIÓN DE COMPONENTES
// ==========================================================================

// Loader Global
function initGlobalLoader() {
    const loader = document.getElementById('global-loader');
    if (!loader) return;
    
    window.addEventListener('load', () => {
        setTimeout(() => {
            loader.style.opacity = '0';
            setTimeout(() => {
                loader.style.display = 'none';
                loader.style.opacity = '1';
            }, 300);
        }, 500);
    });
}

// Toggle Password
function initPasswordToggle() {
    document.addEventListener('click', (e) => {
        const toggleBtn = e.target.closest('[data-toggle-password]');
        if (!toggleBtn) return;
        
        const targetId = toggleBtn.getAttribute('data-target');
        const input = targetId ? document.getElementById(targetId) : 
                    toggleBtn.parentElement.querySelector('input[type="password"], input[type="text"]');
        
        if (input) {
            const type = input.type === 'password' ? 'text' : 'password';
            input.type = type;
            
            const icon = toggleBtn.querySelector('i');
            if (icon) {
                icon.className = type === 'text' ? 'fas fa-eye-slash' : 'fas fa-eye';
            }
        }
    });
}

// Form Validation
function initFormValidation() {
    document.querySelectorAll('form[data-validate]').forEach(form => {
        form.addEventListener('submit', function(e) {
            const required = this.querySelectorAll('[required]');
            let isValid = true;
            
            required.forEach(input => {
                if (!input.value.trim()) {
                    isValid = false;
                    input.classList.add('border-red-500');
                    
                    if (!input.nextElementSibling?.classList.contains('error-message')) {
                        const error = document.createElement('div');
                        error.className = 'error-message text-red-500 text-xs mt-1';
                        error.textContent = 'Este campo es obligatorio';
                        input.parentNode.appendChild(error);
                    }
                } else {
                    input.classList.remove('border-red-500');
                    const error = input.parentNode.querySelector('.error-message');
                    if (error) error.remove();
                }
            });
            
            if (!isValid) {
                e.preventDefault();
                showNotification('error', 'Por favor, completa todos los campos requeridos');
            }
        });
    });
}

// Tooltips
function initTooltips() {
    document.querySelectorAll('[data-toggle="tooltip"]').forEach(el => {
        el.addEventListener('mouseenter', function() {
            const tooltipText = this.getAttribute('title') || this.getAttribute('data-tooltip');
            if (!tooltipText) return;
            
            const tooltip = document.createElement('div');
            tooltip.className = 'absolute z-50 px-2 py-1 text-xs text-white bg-gray-900 rounded shadow-lg';
            tooltip.textContent = tooltipText;
            tooltip.style.position = 'fixed';
            
            const rect = this.getBoundingClientRect();
            tooltip.style.top = (rect.top - 30) + 'px';
            tooltip.style.left = (rect.left + (rect.width / 2)) + 'px';
            tooltip.style.transform = 'translateX(-50%)';
            
            document.body.appendChild(tooltip);
            this.tooltipElement = tooltip;
        });
        
        el.addEventListener('mouseleave', function() {
            if (this.tooltipElement) {
                this.tooltipElement.remove();
            }
        });
    });
}

// Modals
function initModals() {
    // Abrir modales
    document.querySelectorAll('[data-toggle="modal"]').forEach(button => {
        button.addEventListener('click', function() {
            const target = this.getAttribute('data-target');
            const modal = document.getElementById(target);
            
            if (modal) {
                modal.classList.remove('hidden');
                modal.classList.add('flex');
                
                // Focus en el primer input
                const input = modal.querySelector('input, select, textarea');
                if (input) setTimeout(() => input.focus(), 100);
            }
        });
    });
    
    // Cerrar modales
    document.addEventListener('click', (e) => {
        // Botón cerrar
        if (e.target.closest('[data-dismiss="modal"]')) {
            const modal = e.target.closest('.modal');
            if (modal) {
                modal.classList.add('hidden');
                modal.classList.remove('flex');
            }
        }
        
        // Clic fuera del modal
        if (e.target.classList.contains('modal')) {
            e.target.classList.add('hidden');
            e.target.classList.remove('flex');
        }
    });
    
    // Cerrar con ESC
    document.addEventListener('keydown', (e) => {
        if (e.key === 'Escape') {
            document.querySelectorAll('.modal:not(.hidden)').forEach(modal => {
                modal.classList.add('hidden');
                modal.classList.remove('flex');
            });
        }
    });
}

// Menu Responsive
function initResponsiveMenu() {
    const menuToggle = document.querySelector('[data-toggle="menu"]');
    const menu = document.querySelector('[data-menu]');
    
    if (menuToggle && menu) {
        menuToggle.addEventListener('click', () => {
            const isHidden = menu.classList.contains('hidden');
            
            if (isHidden) {
                menu.classList.remove('hidden');
            } else {
                menu.classList.add('hidden');
            }
            
            // Cambiar icono
            const icon = menuToggle.querySelector('i');
            if (icon) {
                icon.className = isHidden ? 'fas fa-times' : 'fas fa-bars';
            }
        });
        
        // Cerrar al hacer clic fuera
        document.addEventListener('click', (e) => {
            if (!menu.contains(e.target) && !menuToggle.contains(e.target) && !menu.classList.contains('hidden')) {
                menu.classList.add('hidden');
                const icon = menuToggle.querySelector('i');
                if (icon) icon.className = 'fas fa-bars';
            }
        });
    }
}

// ==========================================================================
// ALPINEJS COMPONENTS
// ==========================================================================

// Componente de búsqueda
Alpine.data('search', () => ({
    query: '',
    results: [],
    loading: false,
    
    async search() {
        if (this.query.length < 2) {
            this.results = [];
            return;
        }
        
        this.loading = true;
        try {
            const response = await fetch(`/api/search?q=${encodeURIComponent(this.query)}`);
            if (response.ok) {
                this.results = await response.json();
            }
        } catch (error) {
            console.error('Search error:', error);
            this.results = [];
        } finally {
            this.loading = false;
        }
    },
    
    clear() {
        this.query = '';
        this.results = [];
    }
}));

// Componente de upload de archivos
Alpine.data('fileUpload', () => ({
    files: [],
    previews: [],
    maxSize: 10 * 1024 * 1024, // 10MB
    
    init() {
        this.$watch('files', (files) => {
            this.previews = [];
            Array.from(files).forEach((file, index) => {
                if (file.size > this.maxSize) {
                    showNotification('error', `El archivo "${file.name}" excede el tamaño máximo de 10MB`);
                    return;
                }
                
                const reader = new FileReader();
                reader.onload = (e) => {
                    this.previews.push({
                        id: index,
                        name: file.name,
                        size: this.formatBytes(file.size),
                        url: e.target.result,
                        type: file.type
                    });
                };
                reader.readAsDataURL(file);
            });
        });
    },
    
    formatBytes(bytes, decimals = 2) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const dm = decimals < 0 ? 0 : decimals;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + ' ' + sizes[i];
    },
    
    removeFile(index) {
        this.files = Array.from(this.files).filter((_, i) => i !== index);
        this.previews = this.previews.filter(p => p.id !== index);
    },
    
    clearAll() {
        this.files = [];
        this.previews = [];
    }
}));

// ==========================================================================
// INICIALIZACIÓN PRINCIPAL
// ==========================================================================
document.addEventListener('DOMContentLoaded', function() {
    console.log('UrbanOSS v4.0 - Sistema cargado');
    
    // Inicializar AlpineJS
    Alpine.start();
    
    // Inicializar componentes
    initGlobalLoader();
    initPasswordToggle();
    initFormValidation();
    initTooltips();
    initModals();
    initResponsiveMenu();
    
    // Configurar CSRF para fetch
    const csrfToken = UrbanOSS.config.csrfToken;
    if (csrfToken) {
        const originalFetch = window.fetch;
        window.fetch = function(resource, options = {}) {
            if (typeof resource === 'string' && resource.startsWith('/') && 
                options.method && options.method !== 'GET') {
                options.headers = {
                    ...options.headers,
                    'X-CSRF-TOKEN': csrfToken,
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json',
                };
                
                if (options.body && typeof options.body === 'string') {
                    options.headers['Content-Type'] = 'application/json';
                }
            }
            return originalFetch(resource, options);
        };
    }
    
    // Auto-remove flash messages
    setTimeout(() => {
        document.querySelectorAll('.alert').forEach(alert => {
            alert.style.transition = 'opacity 0.5s';
            alert.style.opacity = '0';
            setTimeout(() => alert.remove(), 500);
        });
    }, 5000);
    
    // Configurar confirmación para enlaces de eliminación
    document.addEventListener('click', function(e) {
        const deleteLink = e.target.closest('a[data-confirm-delete], button[data-confirm-delete]');
        if (deleteLink) {
            e.preventDefault();
            const message = deleteLink.getAttribute('data-message') || 
                           '¿Está seguro de eliminar este registro?';
            const form = deleteLink.closest('form');
            const href = deleteLink.getAttribute('href');
            
            confirmDelete(message).then(confirmed => {
                if (confirmed) {
                    if (form) {
                        form.submit();
                    } else if (href) {
                        window.location.href = href;
                    }
                }
            });
        }
    });
    
    // Fix para iconos Font Awesome en Safari
    if (navigator.userAgent.includes('Safari') && !navigator.userAgent.includes('Chrome')) {
        const style = document.createElement('style');
        style.textContent = `
            .fa, .fas, .far, .fal, .fab {
                -webkit-font-smoothing: antialiased !important;
                -moz-osx-font-smoothing: grayscale !important;
            }
        `;
        document.head.appendChild(style);
    }
});

// ==========================================================================
// MANEJO DE ERRORES GLOBALES
// ==========================================================================
window.addEventListener('error', function(event) {
    console.error('Global error:', event.error);
    
    if (UrbanOSS.environment !== 'production') {
        showNotification('error', `Error: ${event.message}`);
    }
});

window.addEventListener('unhandledrejection', function(event) {
    console.error('Unhandled promise rejection:', event.reason);
    
    if (UrbanOSS.environment !== 'production') {
        showNotification('error', `Promise error: ${event.reason.message || event.reason}`);
    }
});

// ==========================================================================
// EXPORTAR FUNCIONES GLOBALES
// ==========================================================================
window.UrbanOSS = UrbanOSS;
window.formatCurrency = UrbanOSS.utils.formatCurrency;
window.formatDate = UrbanOSS.utils.formatDate;

// Inicializar cuando el DOM esté listo
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        // Ya está manejado arriba
    });
} else {
    // DOM ya está listo
    Alpine.start();
}